/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
#include <memory>
#include <string>

#include "libepp_nicbr.H"

#include "DomainDeleteTest.H"
#include "DomainDelete.H"
#include "FileUtil.H"
#include "IoException.H"
#include "XmlException.H"

#include "StrUtil.H"

LIBEPP_NICBR_NS_USE

CPPUNIT_TEST_SUITE_REGISTRATION(DomainDeleteTest);

DomainDeleteTest::DomainDeleteTest() {}

DomainDeleteTest::~DomainDeleteTest() {}

void DomainDeleteTest::setUp() {}

void DomainDeleteTest::tearDown() {}

void DomainDeleteTest::set_xml_template_test()
{
	set_xml_template_test(true);
	set_xml_template_test(false);
}

void DomainDeleteTest::set_xml_template_test(bool withLaunchExtension)
{
	string to_be_parsed =
		"<?xml version='1.0' encoding='UTF-8' standalone='no'?>"
		"<epp xmlns='urn:ietf:params:xml:ns:epp-1.0' "
		"xmlns:xsi='http://www.w3.org/2001/XMLSchema-instance' "
		"xsi:schemaLocation='urn:ietf:params:xml:ns:epp-1.0 "
		"epp-1.0.xsd'>"
		"<command>"
		"<delete>"
		"<domain:delete "
		"xmlns:domain='urn:ietf:params:xml:ns:domain-1.0' "
		"xsi:schemaLocation='urn:ietf:params:xml:ns:domain-1.0 "
		"domain-1.0.xsd'>"
		"<domain:name>$(name)$</domain:name>"
		"</domain:delete>"
		"</delete>"
		"$(ext_begin)$"
		"$(launch_ext)$"
		"$(ext_end)$"
		"$(clTRID)$"
		"</command>"
		"</epp>";

	DomainDelete domain_delete;
	DomainDeleteCmd* domain_delete_cmd = domain_delete.get_command();
	domain_delete_cmd->set_name("asultimasnuvensazuisnoceudaalameda"
	                            "principal.org");

	if (withLaunchExtension) {
		LaunchDeleteCmd launch;
		launch.set_phase(LaunchPhase::SUNRISE);
		launch.set_applicationId("abc123");
		domain_delete_cmd->set_launch(launch);
	}

	domain_delete.get_command()->set_clTRID("ABC-12345");
	domain_delete.set_xml_template(to_be_parsed);

	string expected =
		"<?xml version='1.0' encoding='UTF-8' standalone='no'?>"
		"<epp xmlns='urn:ietf:params:xml:ns:epp-1.0' "
		"xmlns:xsi='http://www.w3.org/2001/XMLSchema-instance' "
		"xsi:schemaLocation='urn:ietf:params:xml:ns:epp-1.0 "
		"epp-1.0.xsd'>"
		"<command>"
		"<delete>"
		"<domain:delete "
		"xmlns:domain='urn:ietf:params:xml:ns:domain-1.0' "
		"xsi:schemaLocation='urn:ietf:params:xml:ns:domain-1.0 "
		"domain-1.0.xsd'>"
		"<domain:name>asultimasnuvensazuisnoceudaalamedaprincipal.org"
		"</domain:name>"
		"</domain:delete>"
		"</delete>";

	if (withLaunchExtension) {
		expected += "<extension>";
	}

	if (withLaunchExtension) {
		expected += "<launch:delete "
			"xmlns:launch=\"urn:ietf:params:xml:ns:launch-1.0\">"
			"<launch:phase>sunrise</launch:phase>"
			"<launch:applicationID>abc123</launch:applicationID>"
			"</launch:delete>";
	}

	if (withLaunchExtension) {
		expected += "</extension>";
	}

	expected += "<clTRID>ABC-12345</clTRID>"
		"</command>"
		"</epp>";
  
	CPPUNIT_ASSERT_EQUAL(expected, domain_delete.get_xml());

	// tests reset method
	domain_delete_cmd = domain_delete.get_command();
	domain_delete_cmd->reset();
	CPPUNIT_ASSERT(domain_delete_cmd->get_name() == "");
  
	// parent attribute
	CPPUNIT_ASSERT(domain_delete_cmd->get_clTRID() == "");
}

void DomainDeleteTest::command_test()
{
	DomainDelete domain_delete;
	DomainDeleteCmd *domain_delete_cmd = domain_delete.get_command();
	domain_delete_cmd->set_name("asultimasnuvensazuisnoceudaalameda"
	                            "principal.org");
  
	LaunchDeleteCmd launch;
	launch.set_phase(LaunchPhase::SUNRISE);
	launch.set_applicationId("abc123");
	domain_delete_cmd->set_launch(launch);

	bool exception_caught = false;
	try {
		string xml_template = 
			FileUtil::read_file("../docs/templates/domain_delete.xml");

		domain_delete.get_command()->set_clTRID("ABC-12345");
		domain_delete.set_xml_template(xml_template);

		DomParser parser;
		parser.enable_validation("../docs/schemas");
		parser.parse_command(domain_delete.get_xml());
	} catch (const IoException &e) {
		exception_caught = true;
		printf("\nIO Exception: code [%d] message [%s]",
		       e.get_code(), e.get_msg().c_str());
	} catch (const XmlException &e) {
		exception_caught = true;
		printf("\nXml Exception: code [%d] message [%s] low level message [%s]\n",
		       e.get_code(), e.get_msg().c_str(), e.get_low_level_msg().c_str());
	}

	CPPUNIT_ASSERT(!exception_caught);
}

void DomainDeleteTest::response_test()
{
	string expected =
		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>"
		"<epp xmlns=\"urn:ietf:params:xml:ns:epp-1.0\" "
		"xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" "
		"xsi:schemaLocation=\"urn:ietf:params:xml:ns:epp-1.0 "
		"epp-1.0.xsd\">"
		"<response>"
		"<result code='1000'>"
		"<msg>Command completed successfully</msg>"
		"</result>"
		"<trID>"
		"<clTRID>ABC-12345</clTRID>"
		"<svTRID>54321-XYZ</svTRID>"
		"</trID>"
		"</response>"
		"</epp>";

	DomParser parser;
	parser.enable_validation("../docs/schemas");

	DomainDelete domain_delete;
	CPPUNIT_ASSERT_NO_THROW(domain_delete.set_response(expected, &parser));
	Response* domain_delete_rsp = domain_delete.get_response();

	map <Response::ResultCode, Response::ResultInfo> results;
	map <Response::ResultCode, Response::ResultInfo>::const_iterator r_it;
	results = domain_delete_rsp->get_result_list();
	r_it = results.begin();

	CPPUNIT_ASSERT(r_it != results.end());
	CPPUNIT_ASSERT_EQUAL(Response::OK, r_it->first);

	string result_code =
		StrUtil::to_string("%d", (int)r_it->first);

	string response =
		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>"
		"<epp xmlns=\"urn:ietf:params:xml:ns:epp-1.0\" "
		"xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" "
		"xsi:schemaLocation=\"urn:ietf:params:xml:ns:epp-1.0 "
		"epp-1.0.xsd\">"
		"<response>"
		"<result code='" + result_code + "'>"
		"<msg";
  
	string result_lang = domain_delete_rsp->get_result_lang();
	if (result_lang != "en") {
		response += " lang='" + result_lang + "'";
	}
  
	response += ">" + r_it->second.msg + "</msg>"
		"</result>"
		"<trID>"
		"<clTRID>" + domain_delete_rsp->get_clTRID() + "</clTRID>"
		"<svTRID>" + domain_delete_rsp->get_svTRID() + "</svTRID>"
		"</trID>"
		"</response>"
		"</epp>";
  
	CPPUNIT_ASSERT_EQUAL(expected, response);
}
