/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file  SMD.H
 *  @brief Signed Mark Data (SMD)
 */

#ifndef __SMD_H__
#define __SMD_H__

#include <list>
#include <map>
#include <string>

#include "CommonData.H"

using std::list;
using std::make_pair;
using std::map;
using std::pair;
using std::string;

LIBEPP_NICBR_NS_BEGIN

/// EPP SMDHolder Class
class SMDHolder
{
public:
	/// EPP SMDHolder::Type Class
	class Type
	{
	public:
		/// Possible types of holder
		enum Value {
			NONE,
			OWNER,
			ASSIGNEE,
			LICENSEE
		};

		/// Convert type into text
		/*
		  @param type holder type
		  @return text representation of the type
		*/
		static string toStr(const Value type) {
			switch(type) {
			case NONE:
				break;
			case OWNER:
				return "owner";
			case ASSIGNEE:
				return "assignee";
			case LICENSEE:
				return "licensee";
			}

			return "";
		}

		/// Convert a text into enum
		/*
		  @param typeStr text based type
		  @return enum representation of the type
		*/
		static Value fromStr(const string typeStr) {
			if (typeStr == "owner") {
				return OWNER;
			} else if (typeStr == "assignee") {
				return ASSIGNEE;
			} else if (typeStr == "licensee") {
				return LICENSEE;
			}

			return NONE;
		}
	};

	/// Default constructor
	SMDHolder()
	{
		reset();
	}

	/// Sets the name of the holder
	/**
	 * @param name Name of the holder
	 */
	void set_name(const string &name) { _name = name; }

	/// Returns the name of the holder
	/**
	 * @return name of the holder
	 */
	string get_name() const { return _name; }

	/// Sets the name of the organization holder of the mark
	/**
	 * @param org name of the organization holder of the mark
	 */
	void set_org(const string &org) { _org = org; }

	/// Returns the name of the organization holder of the mark
	/**
	 * @return name of the organization holder of the mark
	 */
	string get_org() const { return _org; }

	/// Sets the address information of the holder of a mark
	/**
	 * @param postalInfo Address information of the holder of a mark
	 */
	void set_postalInfo(const PostalInfo &postalInfo) { _postalInfo = postalInfo; }

	/// Returns the address information of the holder of a mark
	/**
	 * @return address information of the holder of a mark
	 */
	PostalInfo get_postalInfo() const { return _postalInfo; }

	/// Sets the organization's voice telephone number
	/**
	 * @param voice organization's voice telephone number
	 */
	void set_voice(const string &voice) { _voice = voice; }

	/// Returns the organization's voice telephone number
	/**
	 * @return organization's voice telephone number
	 */
	string get_voice() const { return _voice; }

	/// Sets the organization's facsimile telephone number
	/**
	 * @param fax organization's facsimile telephone number
	 */
	void set_fax(const string &fax) { _fax = fax; }

	/// Returns the organization's facsimile telephone number
	/**
	 * @return organization's facsimile telephone number
	 */
	string get_fax() const { return _fax; }

	/// Sets the contact's email address
	/**
	 * @param email contact's email address
	 */
	void set_email(const string &email) { _email = email; }

	/// Returns the contact's email address
	/**
	 * @return email contact's email address
	 */
	string get_email() const { return _email; }

	/// Reset object attributes
	void reset()
	{
		_name.clear();
		_org.clear();
		_postalInfo.reset();
		_voice.clear();
		_fax.clear();
		_email.clear();
	}

private:
	/// Name of the holder
	string _name;

	/// Name of the organization holder of the mark
	string _org;

	/// Address information of the holder of a mark
	PostalInfo _postalInfo;

	/// Organization's voice telephone number
	string _voice;

	/// Organization's facsimile telephone number
	string _fax;

	/// Email address of the holder
	string _email;
};

/// EPP SMDContact Class
class SMDContact
{
public:
	/// EPP SMDContact::Type Class
	class Type
	{
	public:
		/// Possible types of contacts
		enum Value {
			NONE,
			OWNER,
			AGENT,
			THIRD_PARTY
		};

		/// Convert type into text
		/*
		  @param type contact type
		  @return text representation of the type
		*/
		static string toStr(const Value type) {
			switch(type) {
			case NONE:
				break;
			case OWNER:
				return "owner";
			case AGENT:
				return "agent";
			case THIRD_PARTY:
				return "thirdparty";
			}

			return "";
		}

		/// Convert a text into enum
		/*
		  @param typeStr text based type
		  @return enum representation of the type
		*/
		static Value fromStr(const string typeStr) {
			if (typeStr == "owner") {
				return OWNER;
			} else if (typeStr == "agent") {
				return AGENT;
			} else if (typeStr == "thirdparty") {
				return THIRD_PARTY;
			}

			return NONE;
		}
	};

	/// Default constructor
	SMDContact()
	{
		reset();
	}

	/// Sets the name of the responsible person
	/**
	 * @param name name of the responsible person
	 */
	void set_name(const string &name) { _name = name; }

	/// Returns the name of the responsible person
	/**
	 * @return name of the responsible person
	 */
	string get_name() const { return _name; }

	/// Sets the name of the organization of the contact
	/**
	 * @param org name of the organization of the contact
	 */
	void set_org(const string &org) { _org = org; }

	/// Sets the name of the organization of the contact
	/**
	 * @return name of the organization of the contact
	 */
	string get_org() const { return _org; }

	/// Sets the address information of the contact
	/**
	 * @param postalInfo address information of the contact
	 */
	void set_postalInfo(const PostalInfo &postalInfo) { _postalInfo = postalInfo; }

	/// Returns the address information of the contact
	/**
	 * @return address information of the contact
	 */
	PostalInfo get_postalInfo() const { return _postalInfo; }

	/// Sets the contact's voice telephone number
	/**
	 * @param voice contact's voice telephone number
	 */
	void set_voice(const string &voice) { _voice = voice; }

	/// Returns the contact's voice telephone number
	/**
	 * @return contact's voice telephone number
	 */
	string get_voice() const { return _voice; }

	/// Sets the contact's facsimile telephone number
	/**
	 * @param fax contact's facsimile telephone number
	 */
	void set_fax(const string &fax) { _fax = fax; }

	/// Returns the contact's facsimile telephone number
	/**
	 * @return contact's facsimile telephone number
	 */
	string get_fax() const { return _fax; }

	/// Sets the contact's email address
	/**
	 * @param email contact's email address
	 */
	void set_email(const string &email) { _email = email; }

	/// Returns the contact's email address
	/**
	 * @return contact's email address
	 */
	string get_email() const { return _email; }

	/// Reset object attributes
	void reset()
	{
		_name.clear();
		_org.clear();
		_postalInfo.reset();
		_voice.clear();
		_fax.clear();
		_email.clear();
	}

private:
	/// Name of the responsible person
	string _name;

	/// Name of the organization of the contact
	string _org;

	/// Address information of the contact
	PostalInfo _postalInfo;

	/// Contact's voice telephone number
	string _voice;

	/// Contact's facsimile telephone number
	string _fax;

	/// Contact's email address
	string _email;
};

/// EPP SMDTrademark Class
class SMDTrademark
{
public:
	/// Default constructor
	SMDTrademark()
	{
		reset();
	}

	/// Sets the identifier of the mark
	/**
	 * @param id identifier of the mark
	 */
	void set_id(const string &id) { _id = id; }

	/// Returns the identifier of the mark
	/**
	 * @return identifier of the mark
	 */
	string get_id() const { return _id; }

	/// Sets the mark text string
	/**
	 * @param markName Mark text string
	 */
	void set_markName(const string &markName) { _markName = markName; }

	/// Returns the mark text string
	/**
	 * @return markName Mark text string
	 */
	string get_markName() const { return _markName; }

	/// Sets the information of the holder of the mark
	/**
	 * @param holders information of the holder of the mark
	 */
	void set_holders(const list<pair<SMDHolder::Type::Value, SMDHolder> > &holders)
	{
		_holders = holders;
	}

	/// Add a holder of the mark
	/**
	 * @param type identify the entitlement of the holder
	 * @param holder information of the holder of the mark
	 */
	void add_holder(const SMDHolder::Type::Value type, const SMDHolder &holder)
	{
		_holders.push_back(make_pair(type, holder));
	}

	/// Returns the information of the holder of the mark
	/**
	 * @return information of the holder of the mark
	 */
	list<pair<SMDHolder::Type::Value, SMDHolder> > get_holders() const { return _holders; }

	/// Sets the information of the representative of the mark registration
	/**
	 * @param contacts information of the representative of the mark registration
	 */
	void set_contacts(const map<SMDContact::Type::Value, SMDContact> &contacts)
	{
		_contacts = contacts;
	}
	
	/// Add information of the representative of the mark registration
	/**
	 * @param type contact type
	 * @param contact information of the representative of the mark registration
	 */
	void add_contact(const SMDContact::Type::Value type, const SMDContact &contact)
	{
		_contacts[type] = contact;
	}

	/// Returns the information of the representative of the mark registration
	/**
	 * @return information of the representative of the mark registration
	 */
	map<SMDContact::Type::Value, SMDContact> get_contacts() const { return _contacts; }

	/// Sets the two-character code of the jurisdiction where the
	/// trademark was registered
	/**
	 * @param jurisdiction two-character code of the jurisdiction where
	 * the trademark was registered
	 */
	void set_jurisdiction(const string &jurisdiction) { _jurisdiction = jurisdiction; }

	/// Returns the two-character code of the jurisdiction where the
	/// trademark was registered
	/**
	 * @return two-character code of the jurisdiction where the
	 * trademark was registered
	 */
	string get_jurisdiction() const { return _jurisdiction; }

	/// Sets the Nice Classification class numbers of the mark
	/**
	 * @param classes Nice Classification class numbers of the mark
	 */
	void set_classes(const list<string> &classes) { _classes = classes; }

	/// Add a Nice Classification class numbers of the mark
	/**
	 * @param markClass Nice Classification class numbers of the mark
	 */
	void add_class(const string &markClass) { _classes.push_back(markClass); }

	/// Returns the Nice Classification class numbers of the mark
	/**
	 * @return Nice Classification class numbers of the mark
	 */
	list<string> get_classes() const { return _classes; }

	/// Sets the a-label form of the label that correspond to the
	/// <mark:markName>
	/**
	 * @param labels a-label form of the label that correspond to the
	 * <mark:markName>
	 */
	void set_labels(const list<string> &labels) { _labels = labels; }

	/// Add a a-label form of the label that correspond to the
	/// <mark:markName>
	/**
	 * @param label a-label form of the label that correspond to the
	 * <mark:markName>
	 */
	void add_label(const string &label) { _labels.push_back(label); }

	/// Returns the a-label form of the label that correspond to the
	/// <mark:markName>
	/**
	 * @return a-label form of the label that correspond to the
	 * <mark:markName>
	 */
	list<string> get_labels() const { return _labels; }

	/// Sets the full description of the goods and services mentioned in
	/// the mark registration document
	/**
	 * @param goodsAndServices full description of the goods and
	 * services mentioned in the mark registration document
	 */
	void set_goodsAndServices(const string &goodsAndServices)
	{
		_goodsAndServices = goodsAndServices;
	}

	/// Returns the full description of the goods and services mentioned
	/// in the mark registration document
	/**
	 * @return full description of the goods and services mentioned in
	 * the mark registration document
	 */
	string get_goodsAndServices() const { return _goodsAndServices; }

	/// Sets the trademark application ID registered in the trademark
	/// office
	/**
	 * @param apId trademark application ID registered in the trademark
	 * office
	 */
	void set_apId(const string &apId) { _apId = apId; }

	/// Returns the trademark application ID registered in the trademark
	/// office
	/**
	 * @return trademark application ID registered in the trademark
	 * office
	 */
	string get_apId() const { return _apId; }
	
	/// Sets the date the trademark was applied for
	/**
	 * @param apDate date the trademark was applied for
	 */
	void set_apDate(const string &apDate) { _apDate = apDate; }

	/// Returns the date the trademark was applied for
	/**
	 * @return date the trademark was applied for
	 */
	string get_apDate() const { return _apDate; }

	/// Sets the trademark registration number registered in the
	/// trademark office
	/**
	 * @param regNum trademark registration number registered in the
	 * trademark office
	 */
	void set_regNum(const string &regNum) { _regNum = regNum; }

	/// Returns the trademark registration number registered in the
	/// trademark office
	/**
	 * @return trademark registration number registered in the trademark
	 * office
	 */
	string get_regNum() const { return _regNum; }

	/// Sets the date the trademark was registered
	/**
	 * @param regDate date the trademark was registered
	 */
	void set_regDate(const string &regDate) { _regDate = regDate; }

	/// Returns the date the trademark was registered
	/**
	 * @return date the trademark was registered
	 */
	string get_regDate() const { return _regDate; }

	/// Sets the expiration date of the trademark
	/**
	 * @param exDate expiration date of the trademark
	 */
	void set_exDate(const string &exDate) { _exDate = exDate; }

	/// Returns the expiration date of the trademark
	/**
	 * @return expiration date of the trademark
	 */
	string get_exDate() const { return _exDate; }

	/// Reset object attributes
	void reset()
	{
		_id.clear();
		_markName.clear();
		_holders.clear();
		_contacts.clear();
		_jurisdiction.clear();
		_classes.clear();
		_labels.clear();
		_goodsAndServices.clear();
		_apId.clear();
		_apDate.clear();
		_regNum.clear();
		_regDate.clear();
		_exDate.clear();
	}

private:
	/// Identifier of the mark
	string _id;

	/// Mark text string
	string _markName;

	/// Information of the holder of the mark
	list<pair<SMDHolder::Type::Value, SMDHolder> > _holders;

	/// Information of the representative of the mark registration
	map<SMDContact::Type::Value, SMDContact> _contacts;

	/// Two-character code of the jurisdiction where the trademark was
	/// registered
	string _jurisdiction;

	/// Nice Classification class numbers of the mark as defined in the
	/// Nice List of Classes -
	/// http://www.wipo.int/classifications/nivilo/nice/index.htm
	list<string> _classes;

	/// A-label form of the label that correspond to the <mark:markName>
	list<string> _labels;

	/// Full description of the goods and services mentioned in the mark
	/// registration document
	string _goodsAndServices;

	/// Trademark application ID registered in the trademark office
	string _apId;

	/// Date the trademark was applied for
	string _apDate;

	/// Trademark registration number registered in the trademark office
	string _regNum;

	/// Date the trademark was registered
	string _regDate;

	/// Expiration date of the trademark
	string _exDate;
};

/// EPP SMDProtection Class
class SMDProtection
{
public:
	/// Default constructor
	SMDProtection()
	{
		reset();
	}

	/// Sets the two-character code of the country in which the mark is
	/// protected
	/**
	 * @param cc two-character code of the country in which the mark is
	 * protected
	 */
	void set_cc(const string &cc) { _cc = cc; }

	/// Returns the two-character code of the country in which the mark
	/// is protected
	/**
	 * @return two-character code of the country in which the mark is
	 * protected
	 */
	string get_cc() const { return _cc; }

	/// Sets the name of a city, state, province or other geographic
	/// region of <mark:country> in which the mark is protected
	/**
	 * @param region name of a city, state, province or other geographic
	 * region of <mark:country> in which the mark is protected
	 */
	void set_region(const string &region) { _region = region; }

	/// Returns the name of a city, state, province or other geographic
	/// region of <mark:country> in which the mark is protected
	/**
	 * @return name of a city, state, province or other geographic
	 * region of <mark:country> in which the mark is protected
	 */
	string get_region() const { return _region; }

	/// Sets the two-character code of the countries of the ruling
	/**
	 * @param rulings two-character code of the countries of the ruling
	 */
	void set_rulings(const list<string> &rulings) { _rulings = rulings; }

	/// Add a two-character code of the country of the ruling
	/**
	 * @param ruling two-character code of the country of the ruling
	 */
	void add_ruling(const string &ruling) { _rulings.push_back(ruling); }

	/// Returns the two-character code of the countries of the ruling
	/**
	 * @return two-character code of the countries of the ruling
	 */
	list<string> get_rulings() const { return _rulings; }

	/// Reset object attributes
	void reset()
	{
		_cc.clear();
		_region.clear();
		_rulings.clear();
	}

private:
	/// Two-character code of the country in which the mark is protected
	string _cc;

	/// Name of a city, state, province or other geographic region of
	/// <mark:country> in which the mark is protected
	string _region;

	/// The two-character code of the countries of the ruling
	list<string> _rulings;
};

/// EPP SMDTreatyOrStatute Class
class SMDTreatyOrStatute
{
public:
	/// Default constructor
	SMDTreatyOrStatute()
	{
		reset();
	}

	/// Sets the identifier of the mark
	/**
	 * @param id identifier of the mark
	 */
	void set_id(const string &id) { _id = id; }

	/// Returns the identifier of the mark
	/**
	 * @return identifier of the mark
	 */
	string get_id() const { return _id; }

	/// Sets the mark text string
	/**
	 * @param markName mark text string
	 */
	void set_markName(const string &markName) { _markName = markName; }

	/// Returns the mark text string
	/**
	 * @return mark text string
	 */
	string get_markName() const { return _markName; }

	/// Sets the information of the holder of the mark
	/**
	 * @param holders information of the holder of the mark
	 */
	void set_holders(const list<pair<SMDHolder::Type::Value, SMDHolder> > &holders)
	{
		_holders = holders;
	}

	/// Add a holder of the mark
	/**
	 * @param type identify the entitlement of the holder
	 * @param holder information of the holder of the mark
	 */
	void add_holder(const SMDHolder::Type::Value type, const SMDHolder &holder)
	{
		_holders.push_back(make_pair(type, holder));
	}

	/// Returns information of the holder of the mark
	/**
	 * @return information of the holder of the mark
	 */
	list<pair<SMDHolder::Type::Value, SMDHolder> > get_holders() const { return _holders; }

	/// Sets the information of the representative of the mark
	/// registration
	/**
	 * @param contacts information of the representative of the mark
	 * registration
	 */
	void set_contacts(const	map<SMDContact::Type::Value, SMDContact> &contacts)
	{
		_contacts = contacts;
	}

	/// Add information of the representative of the mark registration
	/**
	 * @param type contact type
	 * @param contact information of the representative of the mark registration
	 */
	void add_contact(const SMDContact::Type::Value type, const SMDContact &contact)
	{
		_contacts[type] = contact;
	}

	/// Returns the information of the representative of the mark
	/// registration
	/**
	 * @return information of the representative of the mark
	 * registration
	 */
	map<SMDContact::Type::Value, SMDContact> get_contacts() const { return _contacts; }

	/// Sets the countries and region of the country where the mark is
	/// protected
	/**
	 * @param protections countries and region of the country where the
	 * mark is protected
	 */
	void set_protections(const list<SMDProtection> &protections)
	{
		_protections = protections;
	}

	/// Add the country and region of the country where the mark is
	/// protected
	/**
	 * @param protection country and region of the country where the
	 * mark is protected
	 */
	void add_protection(const SMDProtection &protection)
	{
		_protections.push_back(protection);
	}

	/// Returns the countries and region of the country where the mark
	/// is protected
	/**
	 * @return countries and region of the country where the mark is
	 * protected
	 */
	list<SMDProtection> get_protections() const { return _protections; }

	/// Sets the a-label form of the label that correspond to the
	/// <mark:markName>
	/**
	 * @param labels a-label form of the label that correspond to the
	 * <mark:markName>
	 */
	void set_labels(const list<string> &labels) { _labels = labels; }

	/// Add a a-label form of the label that correspond to the
	/// <mark:markName>
	/**
	 * @param label a-label form of the label that correspond to the
	 * <mark:markName>
	 */
	void add_label(const string &label) { _labels.push_back(label); }

	/// Returns the a-label form of the label that correspond to the
	/// <mark:markName>
	/**
	 * @return a-label form of the label that correspond to the
	 * <mark:markName>
	 */
	list<string> get_labels() const { return _labels; }

	/// Sets the full description of the goods and services mentioned in
	/// the mark registration document
	/**
	 * @param goodsAndServices full description of the goods and
	 * services mentioned in the mark registration document
	 */
	void set_goodsAndServices(const string &goodsAndServices)
	{
		_goodsAndServices = goodsAndServices;
	}

	/// Returns the full description of the goods and services mentioned
	/// in the mark registration document
	/**
	 * @return full description of the goods and services mentioned in
	 * the mark registration document
	 */
	string get_goodsAndServices() const { return _goodsAndServices; }

	/// Sets the number of the mark of the treaty or statute
	/**
	 * @param refNum number of the mark of the treaty or statute
	 */
	void set_refNum(const string &refNum) { _refNum = refNum; }

	/// Returns the number of the mark of the treaty or statute
	/**
	 * @return number of the mark of the treaty or statute
	 */
	string get_refNum() const { return _refNum; }

	/// Sets the date of protection of the mark
	/**
	 * @param proDate date of protection of the mark
	 */
	void set_proDate(const string &proDate) { _proDate = proDate; }

	/// Returns the date of protection of the mark
	/**
	 * @return date of protection of the mark
	 */
	string get_proDate() const { return _proDate; }

	/// Sets the title of the treaty or statute
	/**
	 * @param title title of the treaty or statute
	 */
	void set_title(const string &title) { _title = title; }

	/// Returns the title of the treaty or statute
	/**
	 * @return title of the treaty or statute
	 */
	string get_title() const { return _title; }

	/// Sets the execution date of the treaty or statute
	/**
	 * @param execDate execution date of the treaty or statute
	 */
	void set_execDate(const string &execDate) { _execDate = execDate; }

	/// Returns the execution date of the treaty or statute
	/**
	 * @return execution date of the treaty or statute
	 */
	string get_execDate() const { return _execDate; }

	/// Reset object attributes
	void reset()
	{
		_id.clear();
		_markName.clear();
		_holders.clear();
		_contacts.clear();
		_protections.clear();
		_labels.clear();
		_goodsAndServices.clear();
		_refNum.clear();
		_proDate.clear();
		_title.clear();
		_execDate.clear();
	}

private:
	/// Identifier of the mark
	string _id;

	/// Mark text string
	string _markName;

	/// Information of the holder of the mark
	list<pair<SMDHolder::Type::Value, SMDHolder> > _holders;

	/// Information of the representative of the mark registration
	map<SMDContact::Type::Value, SMDContact> _contacts;

	/// Countries and region of the country where the mark is protected
	list<SMDProtection> _protections;

	/// A-label form of the label that correspond to the <mark:markName>
	list<string> _labels;

	/// Full description of the goods and services mentioned in the mark
	/// registration document
	string _goodsAndServices;

	/// Number of the mark of the treaty or statute
	string _refNum;

	/// Date of protection of the mark
	string _proDate;

	/// Title of the treaty or statute
	string _title;

	/// Execution date of the treaty or statute
	string _execDate;
};

/// EPP SMDCourt Class
class SMDCourt
{
public:
	/// Default constructor
	SMDCourt()
	{
		reset();
	}

	/// Sets the identifier of the mark
	/**
	 * @param id identifier of the mark
	 */
	void set_id(const string &id) { _id = id; }

	/// Returns the identifier of the mark
	/**
	 * @return identifier of the mark
	 */
	string get_id() const { return _id; }

	/// Sets the mark text string
	/**
	 * @param markName mark text string
	 */
	void set_markName(const string &markName) { _markName = markName; }

	/// Returns the mark text string
	/**
	 * @return mark text string
	 */
	string get_markName() const { return _markName; }

	/// Sets the information of the holder of the mark
	/**
	 * @param holders information of the holder of the mark
	 */
	void set_holders(const list<pair<SMDHolder::Type::Value, SMDHolder> > &holders)
	{
		_holders = holders;
	}

	/// Add a holder of the mark
	/**
	 * @param type identify the entitlement of the holder
	 * @param holder information of the holder of the mark
	 */
	void add_holder(const SMDHolder::Type::Value type, const SMDHolder &holder)
	{
		_holders.push_back(make_pair(type, holder));
	}

	/// Returns the information of the holder of the mark
	/**
	 * @return information of the holder of the mark
	 */
	list<pair<SMDHolder::Type::Value, SMDHolder> > get_holders() const { return _holders; }

	/// Sets the information of the representative of the mark
	/// registration
	/**
	 * @param contacts information of the representative of the mark
	 * registration
	 */
	void set_contacts(const map<SMDContact::Type::Value, SMDContact> &contacts)
	{
		_contacts = contacts;
	}

	/// Add information of the representative of the mark registration
	/**
	 * @param type contact type
	 * @param contact information of the representative of the mark registration
	 */
	void add_contact(const SMDContact::Type::Value type, const SMDContact &contact)
	{
		_contacts[type] = contact;
	}

	/// Returns the information of the representative of the mark
	/// registration
	/**
	 * @return information of the representative of the mark
	 * registration
	 */
	map<SMDContact::Type::Value, SMDContact> get_contacts() const { return _contacts; }

	/// Sets the a-label form of the label that correspond to the
	/// <mark:markName>
	/**
	 * @param labels a-label form of the label that correspond to the
	 * <mark:markName>
	 */
	void set_labels(const list<string> &labels) { _labels = labels; }

	/// Add a a-label form of the label that correspond to the
	/// <mark:markName>
	/**
	 * @param label a-label form of the label that correspond to the
	 * <mark:markName>
	 */
	void add_label(const string &label) { _labels.push_back(label); }

	/// Returns the a-label form of the label that correspond to the
	/// <mark:markName>
	/**
	 * @return a-label form of the label that correspond to the
	 * <mark:markName>
	 */
	list<string> get_labels() const { return _labels; }

	/// Sets the full description of the goods and services mentioned in
	/// the mark registration document
	/**
	 * @param goodsAndServices full description of the goods and
	 * services mentioned in the mark registration document
	 */
	void set_goodsAndServices(const string &goodsAndServices)
	{
		_goodsAndServices = goodsAndServices;
	}

	/// Returns the full description of the goods and services mentioned
	/// in the mark registration document
	/**
	 * @return full description of the goods and services mentioned in
	 * the mark registration document
	 */
	string get_goodsAndServices() const { return _goodsAndServices; }

	/// Sets the reference number of the court's opinion
	/**
	 * @param refNum reference number of the court's opinion
	 */
	void set_refNum(const string &refNum) { _refNum = refNum; }

	/// Returns the reference number of the court's opinion
	/**
	 * @return reference number of the court's opinion
	 */
	string get_refNum() const { return _refNum; }

	/// Sets the date of protection of the mark
	/**
	 * @param proDate date of protection of the mark
	 */
	void set_proDate(const string &proDate) { _proDate = proDate; }

	/// Returns the date of protection of the mark
	/**
	 * @return date of protection of the mark
	 */
	string get_proDate() const { return _proDate; }

	/// Sets the two-character code of the country where the court is
	/// located
	/**
	 * @param cc two-character code of the country where the court is
	 * located
	 */
	void set_cc(const string &cc) { _cc = cc; }

	/// Returns the two-character code of the country where the court is
	/// located
	/**
	 * @return two-character code of the country where the court is
	 * located
	 */
	string get_cc() const { return _cc; }

	/// Sets the name of a city, state, province or other geographic
	/// region of <mark:cc> in which the mark is protected
	/**
	 * @param regions name of a city, state, province or other
	 * geographic region of <mark:cc> in which the mark is protected
	 */
	void set_regions(const list<string> &regions) { _regions = regions; }

	/// Add the name of a city, state, province or other geographic
	/// region of <mark:cc> in which the mark is protected
	/**
	 * @param region name of a city, state, province or other geographic
	 * region of <mark:cc> in which the mark is protected
	 */
	void add_region(const string &region) { _regions.push_back(region); }

	/// Returns the name of a city, state, province or other geographic
	/// region of <mark:cc> in which the mark is protected
	/**
	 * @return name of a city, state, province or other geographic
	 * region of <mark:cc> in which the mark is protected
	 */
	list<string> get_regions() const { return _regions; }

	/// Sets the name of the court
	/**
	 * @param courtName name of the court
	 */
	void set_courtName(const string &courtName) { _courtName = courtName; }

	/// Returns the name of the court
	/**
	 * @return name of the court
	 */
	string get_courtName() const { return _courtName; }

	/// Reset object attributes
	void reset()
	{
		_id.clear();
		_markName.clear();
		_holders.clear();
		_contacts.clear();
		_labels.clear();
		_goodsAndServices.clear();
		_refNum.clear();
		_proDate.clear();
		_cc.clear();
		_regions.clear();
		_courtName.clear();
	}

private:
	/// Identifier of the mark
	string _id;

	/// Mark text string
	string _markName;

	/// Information of the holder of the mark
	list<pair<SMDHolder::Type::Value, SMDHolder> > _holders;

	/// Information of the representative of the mark registration
	map<SMDContact::Type::Value, SMDContact> _contacts;

	/// A-label form of the label that correspond to the <mark:markName>
	list<string> _labels;

	/// Full description of the goods and services mentioned in the mark
	/// registration document
	string _goodsAndServices;

	/// Reference number of the court's opinion
	string _refNum;

	/// Date of protection of the mark
	string _proDate;

	/// Two-character code of the country where the court is located
	string _cc;

	/// Name of a city, state, province or other geographic region of
	/// <mark:cc> in which the mark is protected
	list<string> _regions;

	/// Name of the court
	string _courtName;
};

/// EPP SMDMark Class
class SMDMark
{
public:
	/// Default constructor
	SMDMark()
	{
		reset();
	}

	/// Sets the list of trademarks
	/**
	 * @param trademarks list of trademarks
	 */
	void set_trademarks(const list<SMDTrademark> &trademarks) { _trademarks = trademarks; }

	/// Add a trademark
	/**
	 * @param trademark trademark
	 */
	void add_trademark(const SMDTrademark &trademark) { _trademarks.push_back(trademark); }

	/// Returns the list of trademarks
	/**
	 * @return list of trademarks
	 */
	list<SMDTrademark> get_trademarks() const { return _trademarks; }

	/// Sets the list of treaty or statutes
	/**
	 * @param treatyOrStatute list of treaty or statutes
	 */
	void set_treatyOrStatutes(const list<SMDTreatyOrStatute> &treatyOrStatute)
	{
		_treatyOrStatutes = treatyOrStatute;
	}

	/// Add a treaty or statute
	/**
	 * @param treatyOrStatute treaty or statute
	 */
	void add_treatyOrStatute(const SMDTreatyOrStatute &treatyOrStatute)
	{
		_treatyOrStatutes.push_back(treatyOrStatute);
	}

	/// Returns the list of treaty or statutes
	/**
	 * @return list of treaty or statutes
	 */
	list<SMDTreatyOrStatute> get_treatyOrStatutes() const { return _treatyOrStatutes; }

	/// Sets the list of court
	/**
	 * @param court list of court
	 */
	void set_court(const list<SMDCourt> &court) { _court = court; }

	/// Add a court
	/**
	 * @param court court
	 */
	void add_court(const SMDCourt &court) { _court.push_back(court); }

	/// Returns the list of court
	/**
	 * @return list of court
	 */
	list<SMDCourt> get_court() const { return _court; }

	/// Reset object attributes
	void reset()
	{
		_trademarks.clear();
		_treatyOrStatutes.clear();
		_court.clear();
	}

	bool is_empty() const
	{
		return _trademarks.empty() && _treatyOrStatutes.empty() && _court.empty();
	}

private:
	/// List of trademarks
	list<SMDTrademark> _trademarks;

	/// List of treaty or statutes
	list<SMDTreatyOrStatute> _treatyOrStatutes;

	/// List of court
	list<SMDCourt> _court;
};

/// EPP SMDIssuerInfo Class
class SMDIssuerInfo
{
public:
	/// Default constructor
	SMDIssuerInfo()
	{
		reset();
	}

	/// Sets the identifier of the issuer
	/**
	 * @param id identifier of the issuer
	 */
	void set_id(const string &id) { _id = id; }

	/// Returns the identifier of the issuer
	/**
	 * @return identifier of the issuer
	 */
	string get_id() const { return _id; }

	/// Sets the organization name of the issuer
	/**
	 * @param org organization name of the issuer
	 */
	void set_org(const string &org) { _org = org; }

	/// Returns the organization name of the issuer
	/**
	 * @return organization name of the issuer
	 */
	string get_org() const { return _org; }

	/// Sets the issuer customer support email address
	/**
	 * @param email issuer customer support email address
	 */
	void set_email(const string &email) { _email = email; }

	/// Returns the issuer customer support email address
	/**
	 * @return issuer customer support email address
	 */
	string get_email() const { return _email; }

	/// Sets the HTTP URL of the issuer's site
	/**
	 * @param url HTTP URL of the issuer's site
	 */
	void set_url(const string &url) { _url = url; }

	/// Returns the HTTP URL of the issuer's site
	/**
	 * @return HTTP URL of the issuer's site
	 */
	string get_url() const { return _url; }

	/// Sets the issuer's voice telephone number
	/**
	 * @param voice issuer's voice telephone number
	 */
	void set_voice(const string &voice) { _voice = voice; }

	/// Returns the issuer's voice telephone number
	/**
	 * @return issuer's voice telephone number
	 */
	string get_voice() const { return _voice; }

	/// Reset object attributes
	void reset()
	{
		_id.clear();
		_org.clear();
		_email.clear();
		_url.clear();
		_voice.clear();
	}

private:
	/// Identifier of the issuer
	string _id;

	/// Organization name of the issuer
	string _org;

	/// Issuer customer support email address
	string _email;

	/// HTTP URL of the issuer's site
	string _url;

	/// Issuer's voice telephone number
	string _voice;
};

/// EPP SMD Class
class SMD
{
public:
	/// Default constructor
	SMD()
	{
		reset();
	}

	/// Sets the concatenation of the local identifier, followed by a
	/// hyphen ("-", ASCII value 0x002D), followed by the issuer
	/// identifier
	/**
	 * @param id concatenation of the local identifier, followed by a
	 * hyphen ("-", ASCII value 0x002D), followed by the issuer
	 * identifier
	 */
	void set_id(const string &id) { _id = id; }

	/// Returns the concatenation of the local identifier, followed by a
	/// hyphen ("-", ASCII value 0x002D), followed by the issuer
	/// identifier
	/**
	 * @return concatenation of the local identifier, followed by a
	 * hyphen ("-", ASCII value 0x002D), followed by the issuer
	 * identifier
	 */
	string get_id() const { return _id; }

	/// Sets the information of the issuer of the mark registration
	/**
	 * @param issuerInfo information of the issuer of the mark registration
	 */
	void set_issuerInfo(const SMDIssuerInfo &issuerInfo) { _issuerInfo = issuerInfo; }

	/// Returns the information of the issuer of the mark registration
	/**
	 * @return information of the issuer of the mark registration
	 */
	SMDIssuerInfo get_issuerInfo() const { return _issuerInfo; }

	/// Sets the creation date and time of the signed mark
	/**
	 * @param notBefore creation date and time of the signed mark
	 */
	void set_notBefore(const string &notBefore) { _notBefore = notBefore; }

	/// Returns the creation date and time of the signed mark
	/**
	 * @return creation date and time of the signed mark
	 */
	string get_notBefore() const { return _notBefore; }

	/// Sets the expiration date and time of the signed mark
	/**
	 * @param notAfter expiration date and time of the signed mark
	 */
	void set_notAfter(const string &notAfter) { _notAfter = notAfter; }

	/// Returns the expiration date and time of the signed mark
	/**
	 * @return expiration date and time of the signed mark
	 */
	string get_notAfter() const { return _notAfter; }

	/// Sets the mark information
	/**
	 * @param mark mark information
	 */
	void set_mark(const SMDMark &mark) { _mark = mark; }

	/// Returns the mark information
	/**
	 * @return mark information
	 */
	SMDMark get_mark() const { return _mark; }

	/// Sets the XML Signature for the <smd:signedMark>
	/**
	 * @param signature XML Signature for the <smd:signedMark>
	 */
	void set_signature(const string &signature) { _signature = signature; }

	/// Returns the XML Signature for the <smd:signedMark>
	/**
	 * @return XML Signature for the <smd:signedMark>
	 */
	string get_signature() const { return _signature; }

	/// Reset object attributes
	void reset()
	{
		_id.clear();
		_issuerInfo.reset();
		_notBefore.clear();
		_notAfter.clear();
		_mark.reset();
		_signature.clear();
	}

private:
	/// Concatenation of the local identifier, followed by a hyphen
	/// ("-", ASCII value 0x002D), followed by the issuer identifier
	string _id;

	/// Information of the issuer of the mark registration
	SMDIssuerInfo _issuerInfo;

	/// Creation date and time of the signed mark
	string _notBefore;

	/// Expiration date and time of the signed mark
	string _notAfter;

	/// Mark information
	SMDMark _mark;

	/// XML Signature for the <smd:signedMark>
	string _signature;
};

LIBEPP_NICBR_NS_END

#endif // __SMD_H__
