/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file ReverseDSInfo.H
 *  @brief EPP/DNSSEC DS information class for reverse DNS
 */

#ifndef __REVERSE_DSINFO_H__
#define __REVERSE_DSINFO_H__

#include "CommonData.H"
#include "DSInfo.H"

LIBEPP_NICBR_NS_BEGIN

/// Reverse DSInfo Class
class ReverseDSInfo {
public:
	ReverseDSInfo& operator=(const ReverseDSInfo &dsInfo) {
		_ipRange.reset();
		_key_tag = dsInfo._key_tag;
		_algo = dsInfo._algo;
		_digest_type = dsInfo._digest_type;
		_digest = dsInfo._digest;
		return *this;
	}

	ReverseDSInfo(const ReverseDSInfo &dsInfo) {
		_ipRange = dsInfo._ipRange;
		_key_tag = dsInfo._key_tag;
		_algo = dsInfo._algo;
		_digest_type = dsInfo._digest_type;
		_digest = dsInfo._digest;
	}

	/// Default constructor
	ReverseDSInfo()
	{
		this->reset();
	}

	/// Sets ip range
	/**
	   @param ipRange  ip range
	*/
	void set_ipRange(const IpRange &ipRange)
	{
		_ipRange = ipRange;
	}

	/// Returns ip range
	/**
	   @return ip range
	*/
	IpRange get_ipRange() const
	{
		return _ipRange;
	}

	/// Sets key tag
	/**
	   @param key_tag  key_tag
	*/
	void set_key_tag(const unsigned int key_tag)
	{ 
		_key_tag = key_tag;
	}

	/// Returns the key tag
	/**
	   @return key tag
	*/
	unsigned int get_key_tag() const
	{
		return _key_tag;
	}

	/// Sets algorithm
	/**
	   @param algo algorithm
	*/
	void set_algo(const unsigned int algo)
	{ 
		_algo = algo;
	}

	/// Returns the algorithm
	/**
	   @return algorithm
	*/
	unsigned int get_algo() const { return _algo; }

	/// Sets the digest type
	/**
	   @param digest_type digest type
	*/
	void set_digest_type(const unsigned int digest_type)
	{ 
		_digest_type = digest_type;
	}

	/// Returns the digest type
	/**
	   @return digest type
	*/
	unsigned int get_digest_type() const { return _digest_type; }

	/// Sets the digest 
	/**
	   @param digest the digest 
	*/
	void set_digest(const string& digest)
	{ 
		_digest = digest;
	}

	/// Returns the digest
	/**
	   @return digest
	*/
	string get_digest() const
	{
		return _digest;
	}

	/// Returns the xml format
	/**
	   @return xml
	*/
	string get_xml_format(string secDnsVersion = "") const
	{
		string xml = "<ipnetwork:dsData>"
			"<ipnetwork:ipRange version=\"" + 
			StrUtil::esc_xml_markup(_ipRange.get_version()) + "\">"
			"<ipnetwork:startAddress>" + 
			StrUtil::esc_xml_markup(_ipRange.get_ipBegin()) +
			"</ipnetwork:startAddress>"
			"<ipnetwork:endAddress>" + 
			StrUtil::esc_xml_markup(_ipRange.get_ipEnd()) +
			"</ipnetwork:endAddress>"
			"</ipnetwork:ipRange>"
			"<ipnetwork:keyTag>" + StrUtil::to_string("%u", _key_tag) +
			"</ipnetwork:keyTag>"
			"<ipnetwork:alg>"  + StrUtil::to_string("%u", _algo) +
			"</ipnetwork:alg>"
			"<ipnetwork:digestType>" + StrUtil::to_string("%u", _digest_type) +
			"</ipnetwork:digestType>"
			"<ipnetwork:digest>" + StrUtil::esc_xml_markup(_digest) + 
			"</ipnetwork:digest>"
			"</ipnetwork:dsData>";
    
		return xml;
	}

	void reset()
	{
		_ipRange.reset();
		_key_tag = 0;
		_algo = 0;
		_digest_type = 0;
		_digest = "";
	}

private:
	// DS Info attributes
	IpRange _ipRange;
	unsigned int _key_tag;
	unsigned int _algo;
	unsigned int _digest_type;
	string _digest;
};

LIBEPP_NICBR_NS_END

#endif // __REVERSE_DSINFO_H__
