/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file  Launch.H
 *  @brief EPP LaunchCheckRsp Class
 */

#ifndef __LAUNCH_CHECK_RSP_H__
#define __LAUNCH_CHECK_RSP_H__

#include <list>
#include <string>

#include "Launch.H"

using std::list;
using std::string;

LIBEPP_NICBR_NS_BEGIN

/// EPP LaunchCheckRsp Class
class LaunchCheckRsp
{
public:
	/// EPP LaunchCheckRsp::Name Class
	class Name {
	public:
		/// Default constructor
		Name()
		{
			reset();
		}

		/// Sets the fully qualified name of the queried domain name
		/**
		 * @param fqdn fully qualified name of the queried domain name
		 */
		void set_fqdn(const string &fqdn) { _fqdn = fqdn; }

		/// Returns the fully qualified name of the queried domain name
		/**
		 * @return fully qualified name of the queried domain name
		 */
		string get_fqdn() const { return _fqdn; }

		/// Sets the existence of a trademark
		/**
		 * @param exists trademark existence
		 */
		void set_exists(const bool exists) { _exists = exists; }

		/// Returns the existence of a trademark
		/**
		 * @return trademark existence
		 */
		bool exists() const { return _exists; }

		/// Reset object attributes
		void reset()
		{
			_fqdn.clear();
			_exists = false;
		}

	private:
		/// Contains the fully qualified name of the queried domain name
		string _fqdn;

		/// Indicates if a matching trademark exists for the domain name
		bool _exists;
	};

	/// EPP LaunchCheckRsp::Result Class
	class Result
	{
	public:
		/// Default constructor
		Result()
		{
			reset();
		}

		/// Sets the fully qualified name of the queried domain name with
		/// the exists flag
		/**
		 * @param name fully qualified name of the queried domain name with
		 * the exists flag
		 */
		void set_name(const Name &name) { _name = name; }

		/// Returns the fully qualified name of the queried domain name with
		/// the exists flag
		/**
		 * @return fully qualified name of the queried domain name with he
		 * exists flag
		 */
		Name get_name() const { return _name; }

		/// Sets the claim key that MAY be passed to a third-party trademark
		/// validator such as the Trademark Clearinghouse (TMCH) for
		/// querying the information needed to generate a Trademark Claims
		/// Notice
		/**
		 * @param claimKey claim key
		 */
		void set_claimKey(const string &claimKey) { _claimKey = claimKey; }

		/// Returns the claim key that MAY be passed to a third-party trademark
		/// validator such as the Trademark Clearinghouse (TMCH) for
		/// querying the information needed to generate a Trademark Claims
		/// Notice
		/**
		 * @return claim key
		 */
		string get_claimKey() const { return _claimKey; }

		/// Reset object attributes
		void reset()
		{
			_name.reset();
			_claimKey.clear();
		}

	private:
		/// Contains the fully qualified name of the queried domain name
		/// with the exists flag
		Name _name;

		/// Claim key that MAY be passed to a third-party trademark
		/// validator such as the Trademark Clearinghouse (TMCH) for
		/// querying the information needed to generate a Trademark Claims
		/// Notice
		string _claimKey;
	};

	/// Default constructor
	LaunchCheckRsp()
	{
		reset();
	}

	/// Sets the phase of the launch
	/**
	 * @param phase phase of the launch
	 */
	void set_phase(const LaunchPhase &phase) { _phase = phase; }

	/// Returns the phase of the launch
	/**
	 * @return phase of the launch
	 */
	LaunchPhase get_phase() const { return _phase; }

	/// Sets the list of results
	/**
	 * @param results list of results
	 */
	void set_results(const list<Result> &results) { _results = results; }

	/// Returns the list of results
	/**
	 * @return list of results
	 */
	list<Result> get_results() const { return _results; }
	
	/// Add a launch result
	/**
	 * @param result launch result
	 */
	void add_result(const Result &result)
	{
		_results.push_back(result);
	}

	/// Reset object attributes
	void reset()
	{
		_phase.reset();
		_results.clear();
	}

private:
	/// launch phase that SHOULD be "claims"
	LaunchPhase _phase;

	/// List of results
	list<Result> _results;
};

LIBEPP_NICBR_NS_END

#endif // __LAUNCH_CHECK_RSP_H__
