/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file DefRegUpdateCmd.H
 *  @brief EPP DefRegUpdateCmd Class
 */

#ifndef __DEF_REG_UPDATE_CMD_H__
#define __DEF_REG_UPDATE_CMD_H__

#include <set>
#include <string>

#include "libepp_nicbr.H"

#include "Command.H"
#include "CommonData.H"

using std::set;
using std::string;

LIBEPP_NICBR_NS_BEGIN

/// EPP DefRegUpdateCmd Class
class DefRegUpdateCmd : public Command
{
public:
	/// EPP DefRegUpdateCmd::Status Class
	class Status {
	public:
		/// Default constructor
		Status()
		{
			reset();
		}

		/// Constructor used to easy fill all values
		/**
		 * @param status status of the defensive registration object
		 * @param lang status' message language
		 * @param msg status' message
		 */
		Status(const string &status, const string &lang = "", const string &msg = "") :
			_status(status),
			_lang(lang),
			_msg(msg)
		{
		}

		/// Less comparator to avoid duplicated status in a set container
		/**
		 * @param st other status instance
		 * @return true if this status is less than the other or false
		 * otherwise
		 */
		bool operator<(const Status &st) const
		{
			return _status < st._status;
		}

		/// Sets the status of the defensive registration object
		/**
		 * @param status status of the defensive registration object
		 */
		void set_status(const string &status) { _status = status; }

		/// Returns the status of the defensive registration object
		/**
		 * @return status of the defensive registration object
		 */
		string get_status() const { return _status; }

		/// Sets the status' message language
		/**
		 * @param lang status' message language
		 */
		void set_lang(const string &lang) { _lang = lang; }

		/// Returns the status' message language
		/**
		 * @return status' message language
		 */
		string get_lang() const { return _lang; }

		/// Sets the status' message
		/**
		 * @param msg status' message
		 */
		void set_msg(const string &msg) { _msg = msg; }

		/// Returns the status' message
		/**
		 * @return status' message
		 */
		string get_msg() const { return _msg; }

		/// Reset object attributes
		void reset()
		{
			_status.clear();
			_lang.clear();
			_msg.clear();
		}

	private:
		/// Status of the defensive registration object
		string _status;

		/// Language of the status' message
		string _lang;

		/// Message related to the status
		string _msg;
	};

	/// Default constructor
	DefRegUpdateCmd(bool reset = true) : Command(false)
	{
		if (reset) {
			this->reset();
		}
	}

	/// Set defensive registration roid
	/**
	   @param roid   identification of a defensive registration
	*/
	void set_roid(const string &roid) { _roid = roid; }

	/// Returns defensive registration roid
	/**
	   @return identification of a defensive registration
	*/
	string get_roid() const { return _roid; }

	/// Insert a status to be added in the defensive registration object
	/**
	 * @param status defensive registration object new status
	 */
	void insert_status_add(const Status &status) { _status_add.insert(status); }

	/// Returns a set of status to be added in the defensive
	/// registration object
	/**
	 * @return set of defensive registration object status
	 */
	set<Status> get_status_add() const { return _status_add; }

	/// Insert a status to be removed in the defensive registration object
	/**
	 * @param status defensive registration object status to be removed
	 */
	void insert_status_rem(const Status &status) { _status_rem.insert(status); }

	/// Returns the set of status to be removed in the defensive
	/// registration object
	/**
	 * @return set of defensive registration object status
	 */
	set<Status> get_status_rem() const { return _status_rem; }

	/// Sets registrant
	/**
	   @param registrant   registrant identification
	*/
	void set_registrant(const string& registrant) { _registrant = registrant; }

	/// Returns registrant
	/**
	   @return registrant identification
	*/
	string get_registrant() const { return _registrant; }

	/// Sets the Trademark Identifier (ID) associated with the Defensive
	/// Registration Object
	/**
	 * @param id trademark identifier
	 */
	void set_trademark_id(const string &id) { _trademark_id = id; }

	/// Returns the trademark Identifier (ID) associated with the
	/// Defensive Registration Object
	/**
	 * @return trademark identifier
	 */
	string get_trademark_id() const { return _trademark_id; }

	/// Sets the country which issued the Trademark associated with the
	/// Defensive Registration Object
	/**
	 * @param country country
	 */
	void set_trademark_country(const string &country) { _trademark_country = country; }

	/// Returns country which issued the Trademark associated with the
	/// Defensive Registration Object
	/**
	 * @return country
	 */
	string get_trademark_country() const { return _trademark_country; }

	/// Sets date when the Trademark was issued
	/**
	 * @param date date when the Trademark was issued
	 */
	void set_trademark_date(const string &date) { _trademark_date = date; }

	/// Returns the date when the Trademark was issued
	/**
	 * @return date when the Trademark was issued
	 */
	string get_trademark_date() const { return _trademark_date; }

	/// Sets the identifier for the administrator associated with the
	/// Defensive Registration object
	/**
	 * @param contact administrator contact id
	 */
	void set_admin_contact(const string &contact) { _admin_contact = contact; }

	/// Returns the identifier for the administrator associated with the
	/// Defensive Registration object
	/**
	 * @return administrator contact id
	 */
	string get_admin_contact() const { return _admin_contact; }
  
	/// Sets authorization information
	/**
	   @param authInfo   authorization information
	*/
	void set_authInfo(const AuthInfo &authInfo)
	{
		_authInfo = authInfo;
	}

	/// Returns authorization information
	/**
	   @return authorization information
	*/
	AuthInfo get_authInfo() const { return _authInfo; }

	/// Reset object attributes
	void reset()
	{
		Command::reset();
		_roid.clear();
		_status_add.clear();
		_status_rem.clear();
		_registrant.clear();
		_trademark_id.clear();
		_trademark_country.clear();
		_trademark_date.clear();
		_admin_contact.clear();
		_authInfo.reset();
	}

protected:
	/// Repository Object Identifier assigned to the Defensive
	/// Registration object when the object was created
	string _roid;

	/// Set of status to be added to the defensive registration object
	set<Status> _status_add;

	/// Set of status to be removed from the defensive registration
	/// object
	set<Status> _status_rem;

	/// Identifier for the human or organizational social information
	/// (contact) object to be associated with the Defensive
	/// Registration object as the object registrant
	string _registrant;

	/// Trademark Identifier (ID) associated with the Defensive
	/// Registration Object
	string _trademark_id;

	/// Country which issued the Trademark associated with the Defensive
	/// Registration Object
	string _trademark_country;

	/// Date when the Trademark was issued
	string _trademark_date;

	/// Identifier for the administrator associated with the Defensive
	/// Registration object
	string _admin_contact;

	/// Authorization information to be associated with the Defensive
	/// Registration object
	AuthInfo _authInfo;
};

LIBEPP_NICBR_NS_END

#endif // __DEF_REG_UPDATE_CMD_H__
