/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file DefRegCreateCmd.H
 *  @brief EPP DefRegCreateCmd Class
 */

#ifndef __DEF_REG_CREATE_CMD_H__
#define __DEF_REG_CREATE_CMD_H__

#include <string>

#include "libepp_nicbr.H"

#include "Command.H"
#include "CommonData.H"
#include "RegistrationPeriod.H"

using std::string;

LIBEPP_NICBR_NS_BEGIN

/// EPP DefRegCreateCmd Class
class DefRegCreateCmd : public Command
{
public:
	/// Default constructor
	DefRegCreateCmd(bool reset = true) : Command(false)
	{
		if (reset) {
			this->reset();
		}
	}

	/// Sets name of the Defensive Registration object
	/**
	   @param name name of the Defensive Registration object
	*/
	void set_name(const DefRegName &name) { _name = name; }

	/// Returns name of the Defensive Registration object
	/**
	   @return name of the Defensive Registration object
	*/
	DefRegName get_name() const { return _name; }

	/// Sets registrant
	/**
	   @param registrant   registrant identification
	*/
	void set_registrant(const string& registrant) { _registrant = registrant; }

	/// Returns registrant
	/**
	   @return registrant identification
	*/
	string get_registrant() const { return _registrant; }

	/// Sets the Trademark Identifier (ID) associated with the Defensive
	/// Registration Object
	/**
	 * @param id trademark identifier
	 */
	void set_trademark_id(const string &id) { _trademark_id = id; }

	/// Returns the trademark Identifier (ID) associated with the
	/// Defensive Registration Object
	/**
	 * @return trademark identifier
	 */
	string get_trademark_id() const { return _trademark_id; }

	/// Sets the country which issued the Trademark associated with the
	/// Defensive Registration Object
	/**
	 * @param country country
	 */
	void set_trademark_country(const string &country) { _trademark_country = country; }

	/// Returns country which issued the Trademark associated with the
	/// Defensive Registration Object
	/**
	 * @return country
	 */
	string get_trademark_country() const { return _trademark_country; }

	/// Sets date when the Trademark was issued
	/**
	 * @param date date when the Trademark was issued
	 */
	void set_trademark_date(const string &date) { _trademark_date = date; }

	/// Returns the date when the Trademark was issued
	/**
	 * @return date when the Trademark was issued
	 */
	string get_trademark_date() const { return _trademark_date; }

	/// Sets the initial registration period of the Defensive
	/// Registration object
	/**
	 * @param time   amount of time
	 * @param unit   measure unit
	 */
	void set_period(const int time, const string& unit)
	{
		_period.time = time;
		_period.unit = unit;
	}

	/// Returns the initial registration period of the Defensive
	/// Registration object
	/**
	 * @return initial registration period
	 */
	RegistrationPeriod get_period() const { return _period; }

	/// Sets the identifier for the administrator associated with the
	/// Defensive Registration object
	/**
	 * @param contact administrator contact id
	 */
	void set_admin_contact(const string &contact) { _admin_contact = contact; }

	/// Returns the identifier for the administrator associated with the
	/// Defensive Registration object
	/**
	 * @return administrator contact id
	 */
	string get_admin_contact() const { return _admin_contact; }
  
	/// Sets authorization information
	/**
	   @param authInfo   authorization information
	*/
	void set_authInfo(const AuthInfo &authInfo)
	{
		_authInfo = authInfo;
	}

	/// Returns authorization information
	/**
	   @return authorization information
	*/
	AuthInfo get_authInfo() const { return _authInfo; }

	/// Reset object attributes
	void reset()
	{
		Command::reset();
		_name.reset();
		_registrant.clear();
		_trademark_id.clear();
		_trademark_country.clear();
		_trademark_date.clear();
		_period.time = 0;
		_period.unit = "";
		_admin_contact.clear();
		_authInfo.reset();
	}

protected:
	/// Name of the Defensive Registration object to be created
	DefRegName _name;

	/// Identifier for the human or organizational social information
	/// (contact) object to be associated with the Defensive
	/// Registration object as the object registrant
	string _registrant;

	/// Trademark Identifier (ID) associated with the Defensive
	/// Registration Object
	string _trademark_id;

	/// Country which issued the Trademark associated with the Defensive
	/// Registration Object
	string _trademark_country;

	/// Date when the Trademark was issued
	string _trademark_date;

	/// Initial registration period of the Defensive Registration object
	RegistrationPeriod _period;

	/// Identifier for the administrator associated with the Defensive
	/// Registration object
	string _admin_contact;

	/// Authorization information to be associated with the Defensive
	/// Registration object
	AuthInfo _authInfo;
};

LIBEPP_NICBR_NS_END

#endif // __DEF_REG_CREATE_CMD_H__
