/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file BrOrgUpdateCmd.H
 *  @brief EPP BrOrgUpdateCmd Class
 */

#ifndef __BR_ORG_UPDATE_CMD_H__
#define __BR_ORG_UPDATE_CMD_H__

#include <string>
#include <map>


#include "libepp_nicbr.H"

#include "ContactUpdateCmd.H"
#include "CommonData.H"

using std::string;
using std::map;

LIBEPP_NICBR_NS_BEGIN

/// EPP BrOrgUpdateCmd Class
class BrOrgUpdateCmd : public ContactUpdateCmd
{
public:
	/// Default constructor
	BrOrgUpdateCmd(bool reset = true) : ContactUpdateCmd(false)
	{
		if (reset) {
			this->reset();
		}
	}

	/// Sets Organization attribute
	/**
	   @param organization   Organization Identifier
	*/
	void set_organization(const string &organization)
	{
		_organization = organization;
	}

	/// Returns the Organization
	/**
	   @return organization attribute
	*/
	string get_organization() const
	{
		return _organization;
	}

	/// Sets Responsible for the organization
	/**
	   @param responsible   Person Responsible
	*/
	void set_responsible(const string &responsible) 
	{
		_responsible = responsible;
		_responsible_f = true;
	}

	/// Returns Responsible for the organization
	/**
	   @return Person Responsible
	*/
	string get_responsible() const { return _responsible; };

	/// Sets the responsible change flag
	/**
	   @param responsible_f responsible change flag
	*/
	void set_responsible_f(const bool responsible_f)
	{
		_responsible_f = responsible_f;
	};
  
	/// Returns the responsible change flag
	/**
	   @return The responsible change flag
	*/
	bool get_responsible_f() const
	{
		return _responsible_f;
	};

	/// Sets expiration date
	/**
	   @param exDate   object expiration date
	*/
	void set_exDate(const string& exDate) { _exDate = exDate; }

	/// Returns expiration date
	/**
	   @return expiration date
	*/
	string get_exDate() const { return _exDate; }

	/// Inserts a Contact into the list of additions
	/**
	   @param id Contact Id
	   @param type Contact Type
	*/
	void insert_contact_add(const string &type, const string &id)
	{
		_contact_list_add[type] = id;
	}

	/// Returns the Contact List of additions
	/**
	   @return Contact List
	*/
	map< string, string, less<string> > get_contact_list_add() const
	{
		return _contact_list_add;
	}

	/// Inserts a Contact into the list of deletions
	/**
	   @param id Contact Id
	   @param type Contact Type
	*/
	void insert_contact_rem(const string &type, const string &id)
	{
		_contact_list_rem[type] = id;
	}

	/// Returns the Contact List
	/**
	   @return Contact List
	*/
	map< string, string, less<string> > get_contact_list_rem() const
	{
		return _contact_list_rem;
	}

	/// Sets suspended by payment flag
	/**
	   @param suspended true if the organization is suspended by payment
	                    or false otherwise
	*/
	void set_suspended(const bool suspended) 
	{ 
		if (suspended) {
			_suspended = SuspendedStatus::SUSPENDED;
		} else {
			_suspended = SuspendedStatus::NORMAL;
		}
	}

	/// Returns suspended by payment flag
	/**
	   @return Suspended flag
	*/
	SuspendedStatus::Value get_suspended() const { return _suspended; }

	/// Sets the organization type
	/**
	   @param type organization type
	 */
	void set_type(const string &type)
	{
		_type = type;
	}

	/// Returns the organization type
	/**
	   @return organization type
	 */
	string get_type() const
	{
		return _type;
	}

	/////////////////////////////////////////////
	// LACNIC Org extension methods

	/// Sets the EPP Status
	/**
	   @param epp_status EPP Status
	 */
	void set_epp_status(const string &epp_status)
	{
		_epp_status = epp_status;
	}

	/// Returns the EPP Status
	/**
	   @return EPP Status
	 */
	string get_epp_status() const
	{
		return _epp_status;
	}

	/// Sets the EPP password
	/**
	   @param password EPP password
	 */
	void set_epp_password(const string &password)
	{
		_epp_password = password;
	}

	/// Returns the EPP password
	/**
	   @return EPP password
	 */
	string get_epp_password() const
	{
		return _epp_password;
	}

	/// Inserts an EPP IP or range to be added that will be allowed in
	/// the EPP server
	/*
	  @param ip IP or range
	 */
	void insert_epp_ip_add(const string &ip)
	{
		_epp_ips_add.push_back(ip);
	}

	/// Returns list of IPs or ranges to be added that will be allowed
	/// in the EPP server
	/*
	  @return list of EPP IPs or ranges
	 */
	vector<string> get_epp_ips_add() const
	{
		return _epp_ips_add;
	}

	/// Inserts an EPP IP or range to be removed that will be allowed in
	/// the EPP server
	/*
	  @param ip IP or range
	 */
	void insert_epp_ip_rem(const string &ip)
	{
		_epp_ips_rem.push_back(ip);
	}

	/// Returns list of IPs or ranges to be removed that will be allowed
	/// in the EPP server
	/*
	  @return list of EPP IPs or ranges
	 */
	vector<string> get_epp_ips_rem() const
	{
		return _epp_ips_rem;
	}

	/// Inserts a renewal type to be added
	/*
	  @param type renewal type
	 */
	void insert_renewal_type_add(const string &type)
	{
		_renewal_types_add.push_back(type);
	}

	/// Returns the renewal types to be added
	/*
	  @return renewal types
	 */
	vector<string> get_renewal_types_add() const
	{
		return _renewal_types_add;
	}

	/// Inserts a renewal type to be removed
	/*
	  @param type renewal type
	 */
	void insert_renewal_type_rem(const string &type)
	{
		_renewal_types_rem.push_back(type);
	}

	/// Returns the renewal types to be removed
	/*
	  @return renewal types
	 */
	vector<string> get_renewal_types_rem() const
	{
		return _renewal_types_rem;
	}

	/// Sets the organization resources class
	/*
	  @param resources_class for now can be "all-resources" or "non-legacy-only"
	 */
	void set_resources_class(const string &resources_class)
	{
		_resources_class = resources_class;
	}

	/// Returns the organization resources class
	/*
	  @return organization resources class
	 */
	string get_resources_class() const
	{
		return _resources_class;
	}

	/// Sets the legacy organization password
	/*
	  @param password legacy organization password
	 */
	void set_password(const string &password)
	{
		_password = password;
	}

	/// Returns the legacy organization password
	/*
	  @return legacy organization password
	 */
	string get_password() const
	{
		return _password;
	}

	/// reset attributes
	void reset()
	{
		ContactUpdateCmd::reset();
		_organization = "";
		_contact_list_add.clear();
		_contact_list_rem.clear();
		_responsible = "";
		_responsible_f = false;
		_exDate = "";
		_suspended = SuspendedStatus::UNDEFINED;
		_type = "";
		_epp_status = "";
		_epp_password = "";
		_epp_ips_add.clear();
		_epp_ips_rem.clear();
		_renewal_types_add.clear();
		_renewal_types_rem.clear();
		_resources_class = "";
		_password = "";
	}
  
protected:
	/// map with organization attribute
	string _organization;

	/// Person responsible for the organization
	string _responsible;

	/// Responsible change flag
	bool _responsible_f;

	/// date and  time identifying the end of  the organization object's
	/// registration period
	string _exDate;

	/// Human Contacts to add
	map<string, string, less<string> > _contact_list_add;

	/// Human Contacts to remove
	map<string, string, less<string> > _contact_list_rem;

	/// Suspend organization by payment
	SuspendedStatus::Value _suspended;

	/////////////////////////////////////////////
	// LACNIC Org extension methods

	/// Define the organization type
	string _type;

	/// Define the EPP Status
	string _epp_status;

	/// EPP password for authentication
	string _epp_password;

	/// IP addresses or ranges to be added that are allowed to connect
	/// to the EPP interface
	vector<string> _epp_ips_add;

	/// IP addresses or ranges to be removed that are allowed to connect
	/// to the EPP interface
	vector<string> _epp_ips_rem;

	/// List of category and characteristics of the organization to be
	/// added
	vector<string> _renewal_types_add;

	/// List of category and characteristics of the organization to be
	/// removed
	vector<string> _renewal_types_rem;

	/// There are some options when categorizing an organization. We can
	/// use all resouces to categorize or only the non legacy
	/// resources. The possible values are "all-resources" and
	/// "non-legacy-only"
	string _resources_class;

	/// Authenticate updates of legacy organizations
	string _password;
};

LIBEPP_NICBR_NS_END
#endif //__BR_ORG_UPDATE_CMD_H__
